/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <algorithm>
#include <cstdlib>
#include <sstream>

#include "CalibrationGetLogicalCoordinateCommand.h"
#include "ICommandExecutor.h"
#include "TouchAreaSubdivision.h"

using namespace LayerManagerCalibration;
using namespace std;

ExecutionResult CalibrationGetLogicalCoordinateCommand::execute(ICommandExecutor* execute)
{
    ExecutionResult result = ExecutionFailed;

    // get device configuration
    InputDeviceConfigurationMap::iterator match =
        execute->getInputDeviceConfigurations()->find(string(m_deviceName));
    if (match != execute->getInputDeviceConfigurations()->end())
    {
        InputDeviceConfiguration* config = match->second;

        // set the name of the subdivision
        Subdivision* subdivision;
        if (config->getSubdivisionForCoordinate(*m_raw, &subdivision,
                                                *m_logical))
        {
            // Set the subdivision name
            m_subdivisionName = subdivision->getName();

            // Scale and translate the logical coordinates if the subdivision
            // is a touch area
            if (subdivision->getType() == Subdivision::TOUCH)
            {
                TouchAreaSubdivision* touchSubdivision =
                    static_cast<TouchAreaSubdivision*>(subdivision);
                uint width, height;
                execute->getResolution(&width, &height);
                m_logical->x = (m_logical->x / float(subdivision->getWidth()) * width)
                             + touchSubdivision->getDisplayOffset().x;
                m_logical->y = (m_logical->y / float(subdivision->getHeight()) * height)
                             + touchSubdivision->getDisplayOffset().y;
            }
        }
        else
        {
            // Return logical coordinates even if no subdivisions are found
            config->getCalibration().transformCoordinate(*m_raw, *m_logical);
        }

        result = ExecutionSuccess;
    }

    return result;
}

const string CalibrationGetLogicalCoordinateCommand::getString()
{
    stringstream description;
    description << "CalibrationGetLogicalCoordinateCommand("
                << "deviceName=" << m_deviceName
                << "raw=(" << m_raw->x << "," << m_raw->y << ")"
                << "subdivisionName=" << m_subdivisionName
                << "logical=(" << m_logical->x << "," << m_logical->y << ")"
                << ")";
    return description.str();
}
